<?php

namespace App\Model;

use App\Helpers\StringOperation;
use Illuminate\Database\Eloquent\Model;
use App\Model\Article\ArticleArchive;
use Carbon\Carbon;
use Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class Article extends Model
{

    public $guarded = ['id'];
    public $archive = ArticleArchive::class;
    public $archivable_field = 'created_at';
    public $dates = ['published_at'];

    public $isEvent = false;

    public $casts = [
        'params' => 'object',
        'ingredients' => 'object', // array
        'steps' => 'object',
        'categories' => 'array',
//        'tips' => 'array',
    ];

    public static $fire_events = true;

    use Searchable;

//    use TranslatableColumn;
    use EnumValues;

    public static function boot()
    {
        parent::boot();

        if (!self::$fire_events) {
            return;
        }

        static::creating(function ($model) {
            if (!$model->user_id) {
                // save current user on new article!.
                $model->user_id = Auth::id();
            }
        });

        static::saving(function ($model) {
            // add archive
            (new $model->archive)->addArchive($model->{$model->archivable_field});

            // if no publish date set, set it now.
            if (!$model->isEvent && !$model->published_at && $model->isActive()) {
                $model->published_at = now();
            }

            if (!$model->has_slug_locked) {
                $model->slug = Str::slug($model->title);
            }
            $model->search = Str::slug(implode(' ', [
                $model->title,
                $model->meta_description,
                $model->category->path ?? '',
                $model->category->name ?? '',
                $model->tags->map->only('name')->flatten()->implode(' '),
                StringOperation::extractText($model->body),
            ]), ' ');
        });

        static::deleting(function ($model) {
            // https://github.com/laravel/framework/issues/2536
            // Laravel delete events won't trigger if we don't have the model instances loaded for each photo.
            $model->photos->each(function ($photo) {
                $photo->delete();
            });
            // this WILL not trigger deleting events: $model->photos()->delete();
        });

    }

    public function user()
    {
        return $this->belongsTo('App\Model\User');
    }


    public function comments()
    {
        return $this->hasMany('App\Model\Article\ArticleComment')->active()->with('user');
    }

    public function reviews()
    {
        return $this->hasMany('App\Model\Review');
    }


    public function recentComments()
    {
        return $this->comments()->with('user')->recent()->active()->limit(100);
    }

    public function category()
    {
        return $this->belongsTo('App\Model\Category');

    }

    public function tags()
    {
        return $this->belongsToMany('App\Model\Tag');
    }

    public function documents()
    {
        return $this->hasmany('App\Model\Article\ArticleDocument'); //->withDefault();
    }


//    public function category()
//    {
//        return $this->categories()->take(1);
//    }

    public function getFirstCategoryName()
    {
        return $this->categories->count() ? $this->categories->first()->name : '';
    }

    public function photo()
    {
        return $this->belongsTo('App\Model\Article\ArticlePhoto');
    }

    public function photos()
    {
        return $this->hasMany('App\Model\Article\ArticlePhoto')->orderBy('priority');
    }

    public function scopeRecent($query)
    {
        $query->orderByDesc('published_at');
    }

    public function scopePriority($query)
    {
        $query->orderBy('priority');
    }

    public function scopeFilteredByDomain($query)
    {
        $query->whereIn('visibility', ['global', config('app.host')]);
    }

    public function scopeActive($query)
    {
        $query->where('status', 'active');
//            ->where('published_at', '<=', now());
    }

    public function scopeFeatured($query)
    {
        $query->where('is_featured', true);
    }


    public function scopeBlog($query)
    {
        $query->where('category_id', config('categories_mapping.articles'));
    }


    public function scopeRecipe($query)
    {
        $query->where('category_id', '!=', config('categories_mapping.articles'));
    }

    public function scopeSelectPartial($query)
    {
        $query->select('id', 'title', 'slug', 'photo_id', 'published_at', 'comment_count', 'params', DB::raw('SUBSTRING(body, 1, 1000) AS body'));
    }


    public function getArchive()
    {
        return (new $this->archive)->orderByDesc('year')->orderBy('month')->get()->groupBy('year');
    }

    public function isPage()
    {
        return $this->category_id == config('categories_mapping.pages');
    }

    public function isRecipe()
    {
        if (!$this->exists && $this->category_id != config('categories_mapping.articles')) {
            return true;
        }

        if (empty($this->category)) {
            return true;
        }
        return $this->category->parent_id == config('categories_mapping.recipes');
    }

    public function isBlog()
    {
        return $this->category_id == config('categories_mapping.articles');
    }

    public function getRecipeUrl()
    {
        return route('site.recipes.show', [$this->slug, $this]);

        return route('site.recipes.show', $this);
    }

    public function getArticleUrl()
    {
        return route('site.articles.show', [$this->slug, $this]);
    }

    public function shouldBeAutosaved()
    {
        return !$this->isActive() && $this->exists && $this->created_at->gt(Carbon::now()->subDays(21));
    }

    public function getUrl($category = null)
    {
        if ($this->isRecipe()) {
            return $this->getRecipeUrl();
        }

        if ($this->isPage()) {
            return route('site.dynamic_page', $this);
        }

        return $this->getArticleUrl();
    }


    public function getDummyImage()
    {
//        return asset('assets/site/images/article/placeholder.jpg');
        return 'data:image/gif;base64,R0lGODlhAQABAIAAAMLCwgAAACH5BAAAAAAALAAAAAABAAEAAAICRAEAOw==';
    }

    public function getStepPhotoUrl($step)
    {
        if (empty($step->filename)) {
            return $this->getDummyImage();
        }
        return StringOperation::getPhotoUrlFromNumber($this->id, 'articles', '') . DIRECTORY_SEPARATOR . 'thumb_' . $step->filename;
    }


    public function getStatusClassAttribute()
    {
        switch ($this->status) {
            case 'pending':
                return 'warning';
            case 'active':
                return 'primary';
            case 'inactive':
                return 'danger';

        }
    }

    public function getPhotoUrl()
    {
        if ($this->photo) {
            return $this->photo->getPhotoUrl();
        }
        return $this->getDummyImage();
    }

    public function getThumbUrl()
    {
        if ($this->photo_id && $this->photo) {
            return $this->photo->getThumbUrl();
        }
        return $this->getDummyImage();
    }

    public function isNew()
    {
        if (!$this->created_at) {
            return false;
        }
        return $this->created_at->gt(Carbon::now()->subDay(2));
    }

    public function isPending()
    {
        return $this->status === 'pending';
    }

    public function isActive()
    {
        return $this->status === 'active';
    }

    public function getSimilar($limit = 3)
    {


//        $product_categories = $this->categories->pluck('id');

        return self::active()
            ->selectPartial()
            ->with('photo')
//            ->whereHas('categories', function ($q) use ($product_categories) {
//            return $q->whereIn('categories.id', $product_categories);
//        })
            ->limit($limit)
            ->where('id', '!=', $this->id)
            ->where('category_id', '=', $this->category_id)
            ->orderByDesc('id')
            ->get();

        return self::active()->limit($limit)->get();
    }

}
