<?php

namespace App\Helpers;

use App\Model\Product;
use App\Model\Size;
use App\Model\Color;

/**
 * Simple shopping cart wrapper
 *
 * Information: the attributes MUST be in the same order when adding or deleting an item
 * (no field sorting due extra performance gain?)
 *
 * Fixes:
 * - don't populate the cart when product doenst exist in database anymore (eg: deleted products)
 * - min checkout
 * -
 *
 *
 */
class Cart
{

    private static $all;
    private static $all_with_data;
    private static $all_grouped_by_vendor; // cache.
    private static $total_price;
    public $currency = 'lei';
//    public $currency = 'RON';
    private $store;
    private $store_key = 'cart';
//    private $product_id = 'product_id';
    private $product_id = 'offer_id';
    private $model = Product\ProductOffer::class;
    private $eager_loading = [
        'product.photo',
        'vendor',
        'vendor.setting',
        'product'
    ];
    private $models_key_values = [
        //'color_id' => [Color::class, 'name'],
    ];
    private $sort_by = 'id';
    protected $fields = [
        'offer_id',
//        'vendor_id',
        'stock',
        //'color_id',
//        'delivery_at',
    ];
    protected $fields_used_for_key = [
        'offer_id',
//        'vendor_id',
    ];
    public $warnings = [
        'stock_exceeded' => false,
        'product_removed' => false,
    ];
    private $shipping_taxes = [];
    public $min_checkout = 0;
    public $check_for_stock = false; // @stock-disable

    private $db_aux = []; // used for storing db related.

    public function __construct()
    {
        // @quick-fix
        $this->min_checkout = config('delivery.transport.min');

        $this->store = session();
    }

    public function all($use_cache = true)
    {
        if (self::$all && $use_cache) {
            return self::$all;
        }

        $data = $this->store->get($this->store_key);
        return self::$all = ($data ? $data : collect());
    }

    public function getKey($item)
    {
        // @todo sort by fields order.
        return $item->only($this->fields_used_for_key)->implode(':');
    }

    public function loadData()
    {
        if (!$this->model) {
            throw new \Exception('Model must be not null');
        }
        if (self::$all_with_data) {
            return self::$all_with_data;
        }

        // we cast evertyhing to int to prevent possible SQL injection
        $items = $this->all();

        // if nothing left, just retrun the empty collectin!
        if (!$items->count()) {
            return self::$all_with_data = $items;
        }

        // execute the eloquent query
        $this->db_aux['products'] = $this->model::whereIn('id', $this->all()->pluck($this->product_id))
            ->with($this->eager_loading)
            // ->orderByRaw("FIELD(id,{$items_filtered->implode(',')})")
            ->orderBy($this->sort_by)
            ->get();

        // load friendly names for each needed keys (eg: color, size attrbiutes IDs)
        foreach ($this->models_key_values as $field => $model) {
            $this->db_aux['values'][$field] = $model[0]::whereIn('id', $this->all()->pluck($field))->pluck($model[1], 'id')->toArray();
        }


        // add some db bits to the items result.
        foreach ($items as $item) {
            // associate friendly names with attribute ids.
            $friendly = [];
            foreach ($this->models_key_values as $key => $value) {
                $friendly[$this->getFriendlyKeyName($key)] = isset($item[$key]) ? $this->db_aux['values'][$key][$item[$key]] : null;
            }

            $productDb = $this->db_aux['products']->find($item[$this->product_id]);

            // only load offers with avai product & vendor.
            if (isset($productDb->vendor->id, $productDb->product->id)) {
                // prepare our final data with all needed stuff.
                $productDb && self::$all_with_data[] = new CartItem($item->merge([
                        'product' => $productDb,
                    ] + $friendly));
            }
        }
        // don't let this empty because we have loops over this and we may fail.
        if (!self::$all_with_data) {
            self::$all_with_data = [];
        }
        return self::$all_with_data;
    }

    public function allWithDB()
    {
        $this->loadData();

        return self::$all_with_data ?: [];
    }

    public function getTotalPrice()
    {
        if (self::$total_price) {
            return self::$total_price;
        }
        $this->loadData();
        $sum = 0;
        foreach (self::$all_with_data as $item) {
            $sum += $item->getTotalPrice();
        }
        return self::$total_price = 0 + $sum;
    }

    public function getShippingTax()
    {
        return $this->allGroupedByVendor()->sum('delivery_tax');
    }

    public function getTotalPriceWithShipping()
    {
        return $this->getTotalPrice() + $this->getShippingTax();
    }

    public function processItem($item, $qty = 1)
    {

        if ($this->check_for_stock && $item['qty'] > $item['stock']) {
            $qty = $item['stock'];
        }

        return $item->only($this->fields)->merge(['qty' => $qty]);
    }

    public function updateItems($item, $qty = +1, $override_qty = false)
    {
        $items = $this->all(false);
        $key = $this->getKey($item);

        // process if item exists
        if ($items->has($key)) {
            $items->transform(function ($current_item, $current_key) use ($item, $key, $qty, $override_qty) {

                if ($current_key == $key) {
                    // increase/decrease qty
                    $current_item['qty'] = $override_qty ? $qty : $current_item['qty'] + $qty;
                    // check if qty exceeds stock
                    if ($this->check_for_stock && $current_item['qty'] > $current_item['stock']) {
                        $current_item['qty'] = $current_item['stock'];
                        $this->warn('stock_exceeded');
                    }
                }
                return $current_item;
            });
        } elseif ($qty > 0) {
            $items->put($key, $this->processItem($item, $qty));
        }

        // update static data and filter empty items.
        self::$all = $items->filter(function ($item) {
            // return only items that remains
            return $item['qty'] > 0;
        });

        // save to our store
        $this->store->put($this->store_key, self::$all);
    }

//    public function processStoc
    public function add($item, $qty = 1)
    {
//        $this->store->forget($this->store_key);
        $this->updateItems($item, $qty);
    }

    public function deleteOne($item)
    {
        $this->updateItems($item, -1);
    }

    public function delete($item)
    {
        $this->updateItems($item, -9999999);
    }

    public function update($item, $qty)
    {
//        $this->store->forget($this->store_key);
        $this->updateItems($item, $qty, true);
    }

    public function count()
    {
        return $this->all()->count();
    }

    public function count_all()
    {
        return $this->all()->reduce(function ($carry, $item) {
            return $carry + $item['qty'];
        });
    }

    public function clear()
    {
        $this->store->forget($this->store_key);
        self::$all = self::$all_with_data = collect();
    }

    public function canCheckout()
    {
        return $this->getTotalPrice() >= $this->min_checkout;
    }

    public function hasStockIssues() {
        return count(array_filter($this->allWithDB(), function($item) {
            return !$item->isEnoughStock();
        }));
    }

    public function differenceForCheckout()
    {
        return $this->min_checkout - $this->getTotalPrice();
    }

    private function warn($error)
    {
        $this->warnings[$error] = true;
    }

    private function getFriendlyKeyName($key)
    {
        return explode('_', $key)[0];
    }

    public function allGroupedByVendor()
    {
        if (self::$all_grouped_by_vendor) {
            return self::$all_grouped_by_vendor;
        }

        return self::$all_grouped_by_vendor = collect($this->allWithDB())
            ->groupBy(function ($item) {
                return $item->product->vendor->id;
            })->map(function ($items) {
                // take vendor data from first product.
                $vendor = $items->first()->product->vendor;

                // calculate total value of products
                $total = $items->reduce(function ($carry, $item) {
                    return $carry + $item->getTotalPrice();
                }, 0);

                // get delivery tax
                $delivery_tax = 0;
                if ($total < ($vendor->setting->delivery_tax_free_after ?? config('delivery.transport.limit'))) {
                    $delivery_tax = $vendor->setting->delivery_tax ?? config('delivery.transport.tax');
                }

                // return needed info
                return (object)[
                    'vendor' => $vendor,
                    'items' => $items,
                    'total' => $total,
                    'delivery_tax' => $delivery_tax,
                    'total_with_tax' =>  0 + $total + $delivery_tax,
                ];
            });
    }

}
